<?php
class Company {
    // Database connection
    private $conn;
    private $table = "companies";

    // Properties
    public $company_id;
    public $parent_id;
    public $company_name;
    public $company_logo;
    public $company_email;
    public $company_subscription;
    public $company_current_accounts;
    public $company_max_acounts;

    // Constructor
    public function __construct($db) {
        $this->conn = $db;
    }
    
    public function readByCompanyId($company_id) {
        $query = "SELECT
            company_id,
            company_name,
            company_logo
        FROM 
            " . $this->table . "
        WHERE
            company_id = :company_id";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":company_id", $company_id);
        $stmt->execute();
        return $stmt;
    }
    
    public function readParentCompanies() {
        $query = "SELECT
            companies.company_id,
            companies.company_name,
            companies.company_logo
            FROM 
                " . $this->table . "
            WHERE
                parent_id IS NULL";

        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt;
    }
    
      public function readAccountById($company_id) {
        $query = "SELECT
            company_id,
            company_name,
            company_logo
        FROM 
            " . $this->table . "
        WHERE
            company_id = :company_id";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":company_id", $company_id);
        $stmt->execute();
        return $stmt;
    }

    // Get admin company names
    public function getAdminCompanies($admin_id) {
        $query = "SELECT 
            companies.company_id, 
            companies.company_name
        FROM    " . $this->table . " 
        INNER JOIN company_admin ON companies.company_id = company_admin.company_id 
        INNER JOIN admins ON company_admin.admin_id = admins.admin_id 
        WHERE user_id = :admin_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":admin_id", $admin_id);
        $stmt->execute();
        return $stmt;
    }

    // Get admin company accounts
    public function getAdminCompanyAccounts($parent_id, $company_id, $admin_id) {
        $query = "SELECT 
            companies.company_id,
            companies.company_name 
        FROM  " . $this->table . " 
        INNER JOIN company_admin ON companies.company_id = company_admin.company_id 
        INNER JOIN admins ON company_admin.admin_id = admins.admin_id 
        WHERE parent_id = :parent_id AND companies.company_id = :company_id AND user_id = :admin_id
        ";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":parent_id", $parent_id);
        $stmt->bindParam(":company_id", $company_id);
        $stmt->bindParam(":admin_id", $admin_id);
        $stmt->execute();
        return $stmt;
    }

    // Get active locations for a given company
    public function getCompanyLocations($company_id) {
        $query = "SELECT 
            location_id, location_name AS name
        FROM locations
        WHERE company_id = :company_id AND active = 1
        ";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":company_id", $company_id);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Get archived locations for a given company
    public function getArchivedLocations($company_id){
        $query = "SELECT 
            location_id, location_name AS name
        FROM locations
        WHERE company_id = :company_id AND active = 0
        ";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":company_id", $company_id);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Get active users for a given company
    public function getActiveUsers($company_id) {
        $query = "SELECT
            users.user_id,
            CONCAT(user_fname, ' ', user_lname) AS name
        FROM 
            users
            INNER JOIN user_company ON users.user_id = user_company.user_id 
        WHERE
            company_id = :company_id
        AND active = 1";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":company_id", $company_id);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Get archived users for a given company
    public function getArchivedUsers($company_id) {
        $query = "SELECT
            users.user_id,
            CONCAT(user_fname, ' ', user_lname) AS name
        FROM 
            users
            INNER JOIN user_company ON users.user_id = user_company.user_id 
        WHERE
            company_id = :company_id
        AND active = 0";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":company_id", $company_id);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }


   // Get child companies for a given parent company
    public function getChildCompanies($parentCompanyId) {
        $query = "SELECT
            company_id,
            company_logo,
            company_name
        FROM 
            " . $this->table . "
        WHERE
            parent_id = :parent_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":parent_id", $parentCompanyId);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }


    
    // Get all 
    public function read1() {
        $query = "SELECT
            company_id,
            company_name,
            company_logo,
            company_email,
            company_subscription,
            company_current_accounts,
            company_max_accounts
        FROM 
            " . $this->table . 
            "";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt;
    }
    
    // Get all companies WHERE parent_id IS NULL
    public function read() {
        $query = "SELECT
            company_id,
            company_name,
            company_logo,
            company_email,
            company_subscription,
            company_current_accounts,
            company_max_accounts
        FROM 
            " . $this->table . "
        WHERE
            parent_id IS NULL";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt;
    }

    // Get company by id
    public function read_single() {
        $query = "SELECT
            company_id,
            company_name,
            company_logo,
            company_email,
            company_subscription,
            company_current_accounts,
            company_max_accounts
        FROM 
            " . $this->table . "
        WHERE
            company_id = :company_id
        ";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":company_id", $this->company_id); // Bind 1st ? to company_id
        $stmt->execute();

        // Fetch the result as assoc_array
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
    
        // One row in API call
        $this->company_id = $row["company_id"]; 
        $this->company_name = $row["company_name"]; 
        $this->company_logo = $row["company_logo"]; 
        $this->company_email = $row["company_email"]; 
        $this->company_subscription = $row["company_subscription"]; 
        $this->company_current_accounts = $row["company_current_accounts"]; 
        $this->company_max_accounts = $row["company_max_accounts"]; 
    }
    
    // Create
    public function create() {
        $query = "INSERT INTO " . $this->table . "
        SET
            company_name = :company_name,
            company_logo = :company_logo,
            company_email = :company_email,
            company_subscription = :company_subscription,
            company_current_accounts = :company_current_accounts,
            company_max_accounts = :company_max_accounts
        ";

        $stmt = $this->conn->prepare($query);

        // Filter data
        $this->company_name = htmlspecialchars(strip_tags($this->company_name));
        $this->company_logo = htmlspecialchars(strip_tags($this->company_logo));
        $this->company_email = htmlspecialchars(strip_tags($this->company_email));
        $this->company_subscription = htmlspecialchars(strip_tags($this->company_subscription));
        $this->company_current_accounts = htmlspecialchars(strip_tags($this->company_current_accounts));
        $this->company_max_accounts = htmlspecialchars(strip_tags($this->company_max_accounts));

        // Bind data
        $stmt->bindParam(":company_name", $this->company_name);
        $stmt->bindParam(":company_logo", $this->company_logo);
        $stmt->bindParam(":company_email", $this->company_email);
        $stmt->bindParam(":company_subscription", $this->company_subscription);
        $stmt->bindParam(":company_current_accounts", $this->company_current_accounts);
        $stmt->bindParam(":company_max_accounts", $this->company_max_accounts);

        // Execute
        if ($stmt->execute()) {
            return true;
        }

        printf("Error: %s", $stmt->error);
        return false;
    }
    
    // Update
    public function update() {
        $query = "UPDATE " . $this->table . "
        SET
            company_name = :company_name,
            company_logo = :company_logo,
            company_email = :company_email,
            company_subscription = :company_subscription,
            company_current_accounts = :company_current_accounts,
            company_max_accounts = :company_max_accounts
        WHERE
            company_id = :company_id
        ";

        $stmt = $this->conn->prepare($query);

        // Filter data
        $this->company_id = htmlspecialchars(strip_tags($this->company_id));
        $this->company_name = htmlspecialchars(strip_tags($this->company_name));
        $this->company_logo = htmlspecialchars(strip_tags($this->company_logo));
        $this->company_email = htmlspecialchars(strip_tags($this->company_email));
        $this->company_subscription = htmlspecialchars(strip_tags($this->company_subscription));
        $this->company_current_accounts = htmlspecialchars(strip_tags($this->company_current_accounts));
        $this->company_max_accounts = htmlspecialchars(strip_tags($this->company_max_accounts));

        // Bind data
        $stmt->bindParam(":company_id", $this->company_id);
        $stmt->bindParam(":company_name", $this->company_name);
        $stmt->bindParam(":company_logo", $this->company_logo);
        $stmt->bindParam(":company_email", $this->company_email);
        $stmt->bindParam(":company_subscription", $this->company_subscription);
        $stmt->bindParam(":company_current_accounts", $this->company_current_accounts);
        $stmt->bindParam(":company_max_accounts", $this->company_max_accounts);

        // Execute
        if ($stmt->execute()) {
            return true;
        }

        printf("Error: %s", $stmt->error);
        return false;
    }
    
    // Delete
    public function delete() {
        $query = "DELETE FROM " . $this->table . " WHERE company_id = :company_id";
        $stmt = $this->conn->prepare($query);

        $this->company_id = htmlspecialchars(strip_tags($this->company_id));
        $stmt->bindParam(":company_id", $this->company_id);

        // Execute
        if ($stmt->execute()) {
            return true;
        }

        printf("Error: %s", $stmt->error);
        return false;
    }
}